<?php
/**
 * API de Gerenciamento - Point Blank Royal Launcher
 * 
 * Sistema completo de controle, monitoramento e estatísticas
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET');
header('Access-Control-Allow-Headers: Content-Type');

// Configuração do banco de dados
$host = 'localhost';
$user = 'seu_usuario';
$pass = 'sua_senha';
$db = 'pointblank_launcher';

$conn = mysqli_connect($host, $user, $pass, $db);

if (!$conn) {
    die(json_encode(['error' => 'Falha na conexão com banco de dados']));
}

mysqli_set_charset($conn, 'utf8mb4');

// Pega ação da requisição
$action = $_GET['action'] ?? $_POST['action'] ?? 'status';

switch ($action) {
    case 'check_user':
        checkUser($conn);
        break;
    
    case 'report_launch':
        reportLaunch($conn);
        break;
    
    case 'report_process':
        reportProcess($conn);
        break;
    
    case 'check_ban':
        checkBan($conn);
        break;
    
    case 'heartbeat':
        heartbeat($conn);
        break;
    
    case 'get_stats':
        getStats($conn);
        break;
    
    default:
        echo json_encode(['error' => 'Ação inválida']);
}

mysqli_close($conn);

/**
 * Verifica se usuário pode usar o launcher
 */
function checkUser($conn) {
    $hwid = $_POST['hwid'] ?? '';
    $ip = $_SERVER['REMOTE_ADDR'];
    $pc_name = $_POST['pc_name'] ?? 'Unknown';
    $username = $_POST['username'] ?? 'Unknown';
    
    if (empty($hwid)) {
        echo json_encode(['allowed' => false, 'reason' => 'HWID inválido']);
        return;
    }
    
    // Registra ou atualiza usuário
    $stmt = $conn->prepare("
        INSERT INTO users (hwid, ip_address, pc_name, username, first_seen, last_seen)
        VALUES (?, ?, ?, ?, NOW(), NOW())
        ON DUPLICATE KEY UPDATE
            ip_address = ?,
            pc_name = ?,
            username = ?,
            last_seen = NOW(),
            total_launches = total_launches + 1
    ");
    $stmt->bind_param("sssssss", $hwid, $ip, $pc_name, $username, $ip, $pc_name, $username);
    $stmt->execute();
    
    // Verifica se está banido
    $stmt = $conn->prepare("
        SELECT id, reason, banned_until, permanent 
        FROM bans 
        WHERE hwid = ? AND (permanent = 1 OR banned_until > NOW())
        ORDER BY created_at DESC
        LIMIT 1
    ");
    $stmt->bind_param("s", $hwid);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($ban = $result->fetch_assoc()) {
        echo json_encode([
            'allowed' => false,
            'banned' => true,
            'reason' => $ban['reason'],
            'permanent' => (bool)$ban['permanent'],
            'banned_until' => $ban['banned_until'],
            'support_url' => 'https://pbroyal.com/suporte'
        ]);
        return;
    }
    
    // Usuário permitido
    echo json_encode([
        'allowed' => true,
        'banned' => false,
        'hwid' => $hwid
    ]);
}

/**
 * Registra abertura do launcher
 */
function reportLaunch($conn) {
    $hwid = $_POST['hwid'] ?? '';
    $ip = $_SERVER['REMOTE_ADDR'];
    
    if (empty($hwid)) return;
    
    $stmt = $conn->prepare("
        INSERT INTO launcher_sessions (hwid, ip_address, started_at)
        VALUES (?, ?, NOW())
    ");
    $stmt->bind_param("ss", $hwid, $ip);
    $stmt->execute();
    
    $session_id = $conn->insert_id;
    
    echo json_encode([
        'success' => true,
        'session_id' => $session_id
    ]);
}

/**
 * Reporta processo detectado
 */
function reportProcess($conn) {
    $hwid = $_POST['hwid'] ?? '';
    $process_name = $_POST['process_name'] ?? '';
    $ip = $_SERVER['REMOTE_ADDR'];
    
    if (empty($hwid) || empty($process_name)) return;
    
    // Registra detecção
    $stmt = $conn->prepare("
        INSERT INTO process_detections (hwid, process_name, ip_address, detected_at)
        VALUES (?, ?, ?, NOW())
    ");
    $stmt->bind_param("sss", $hwid, $process_name, $ip);
    $stmt->execute();
    
    // Verifica se deve banir automaticamente
    $stmt = $conn->prepare("
        SELECT COUNT(*) as count 
        FROM process_detections 
        WHERE hwid = ? AND detected_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)
    ");
    $stmt->bind_param("s", $hwid);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    
    $should_ban = false;
    
    // Bane se detectou hack 3+ vezes em 1 hora
    if ($row['count'] >= 3) {
        $reason = "Uso de programas proibidos: {$process_name}";
        
        $stmt = $conn->prepare("
            INSERT INTO bans (hwid, reason, banned_by, permanent, created_at)
            VALUES (?, ?, 'Sistema', 1, NOW())
        ");
        $stmt->bind_param("ss", $hwid, $reason);
        $stmt->execute();
        
        $should_ban = true;
    }
    
    echo json_encode([
        'success' => true,
        'banned' => $should_ban,
        'detections_count' => $row['count']
    ]);
}

/**
 * Verifica se foi banido (polling)
 */
function checkBan($conn) {
    $hwid = $_POST['hwid'] ?? '';
    
    if (empty($hwid)) {
        echo json_encode(['banned' => false]);
        return;
    }
    
    $stmt = $conn->prepare("
        SELECT reason, banned_until, permanent 
        FROM bans 
        WHERE hwid = ? AND (permanent = 1 OR banned_until > NOW())
        ORDER BY created_at DESC
        LIMIT 1
    ");
    $stmt->bind_param("s", $hwid);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($ban = $result->fetch_assoc()) {
        echo json_encode([
            'banned' => true,
            'reason' => $ban['reason'],
            'permanent' => (bool)$ban['permanent'],
            'banned_until' => $ban['banned_until'],
            'support_url' => 'https://pbroyal.com/suporte'
        ]);
    } else {
        echo json_encode(['banned' => false]);
    }
}

/**
 * Heartbeat - atualiza status online
 */
function heartbeat($conn) {
    $hwid = $_POST['hwid'] ?? '';
    $session_id = $_POST['session_id'] ?? 0;
    
    if (empty($hwid)) return;
    
    // Atualiza última atividade
    $stmt = $conn->prepare("
        UPDATE users 
        SET last_activity = NOW() 
        WHERE hwid = ?
    ");
    $stmt->bind_param("s", $hwid);
    $stmt->execute();
    
    // Atualiza sessão
    if ($session_id > 0) {
        $stmt = $conn->prepare("
            UPDATE launcher_sessions 
            SET last_heartbeat = NOW() 
            WHERE id = ?
        ");
        $stmt->bind_param("i", $session_id);
        $stmt->execute();
    }
    
    echo json_encode(['success' => true]);
}

/**
 * Estatísticas gerais
 */
function getStats($conn) {
    // Total de usuários
    $result = $conn->query("SELECT COUNT(*) as total FROM users");
    $total_users = $result->fetch_assoc()['total'];
    
    // Usuários online (ativos nos últimos 5 minutos)
    $result = $conn->query("
        SELECT COUNT(*) as total 
        FROM users 
        WHERE last_activity > DATE_SUB(NOW(), INTERVAL 5 MINUTE)
    ");
    $online_now = $result->fetch_assoc()['total'];
    
    // Launches hoje
    $result = $conn->query("
        SELECT COUNT(*) as total 
        FROM launcher_sessions 
        WHERE DATE(started_at) = CURDATE()
    ");
    $launches_today = $result->fetch_assoc()['total'];
    
    // Launches esta semana
    $result = $conn->query("
        SELECT COUNT(*) as total 
        FROM launcher_sessions 
        WHERE started_at > DATE_SUB(NOW(), INTERVAL 7 DAY)
    ");
    $launches_week = $result->fetch_assoc()['total'];
    
    // Launches este mês
    $result = $conn->query("
        SELECT COUNT(*) as total 
        FROM launcher_sessions 
        WHERE MONTH(started_at) = MONTH(NOW()) 
        AND YEAR(started_at) = YEAR(NOW())
    ");
    $launches_month = $result->fetch_assoc()['total'];
    
    // Total de banimentos
    $result = $conn->query("SELECT COUNT(*) as total FROM bans");
    $total_bans = $result->fetch_assoc()['total'];
    
    // Detecções de hack hoje
    $result = $conn->query("
        SELECT COUNT(*) as total 
        FROM process_detections 
        WHERE DATE(detected_at) = CURDATE()
    ");
    $hacks_today = $result->fetch_assoc()['total'];
    
    echo json_encode([
        'total_users' => $total_users,
        'online_now' => $online_now,
        'launches' => [
            'today' => $launches_today,
            'week' => $launches_week,
            'month' => $launches_month
        ],
        'bans' => [
            'total' => $total_bans
        ],
        'detections' => [
            'today' => $hacks_today
        ]
    ]);
}
?>
